// --- Logika Inti Sistem Lisensi ---

const KUNCI_RAHASIA = 'KunciSederhanaUntukAplikasiIni123';
const LOKASI_LISENSI = 'dataLisensiAplikasiPOS'; // Nama diubah agar unik untuk aplikasi ini
const LOKASI_VOUCHER_BEKAS = 'voucherDigunakanPOS'; // Nama diubah agar unik

/**
 * Membuat signature/hash untuk validasi.
 * @param {string} data - Data yang akan di-hash.
 * @param {number} panjang - Panjang output hash.
 * @returns {string} Signature dalam format uppercase.
 */
function buatSignature(data, panjang = 8) {
    return CryptoJS.SHA1(data + KUNCI_RAHASIA).toString(CryptoJS.enc.Hex).substring(0, panjang).toUpperCase();
}

/**
 * Membuat ID unik sederhana untuk identifikasi klien.
 * @returns {string} ID Unik 8 karakter.
 */
function buatIdUnikSederhana() {
    const chars = 'ABCDEFGHIJKLMNPQRSTUVWXYZ123456789';
    let result = '';
    for (let i = 0; i < 8; i++) {
        result += chars.charAt(Math.floor(Math.random() * chars.length));
    }
    return result;
}

/**
 * Membuat dan menyimpan lisensi baru untuk pengguna pertama kali.
 * Default masa aktif adalah 30 hari.
 */
function buatLisensiBaru() {
    const idUnik = buatIdUnikSederhana();
    const tgl = new Date();
    let tglKadaluarsa = new Date();
    tglKadaluarsa.setDate(tgl.getDate() + 30); // Lisensi awal 30 hari
    const tglISO = tglKadaluarsa.toISOString();
    const signature = buatSignature(idUnik + tglISO);
    const dataLisensi = {
        idUnik,
        tanggalInstall: tgl.toISOString(),
        tanggalKadaluarsa: tglISO,
        signature,
    };
    localStorage.setItem(LOKASI_LISENSI, JSON.stringify(dataLisensi));
}

/**
 * Memeriksa validitas dan status lisensi yang ada.
 * @returns {object} Objek status lisensi.
 */
function cekLisensi() {
    let data = localStorage.getItem(LOKASI_LISENSI);
    if (!data) {
        buatLisensiBaru();
        data = localStorage.getItem(LOKASI_LISENSI);
    }
    const lisensi = JSON.parse(data);
    const { idUnik, tanggalKadaluarsa, signature } = lisensi;
    const sigHarusnya = buatSignature(idUnik + tanggalKadaluarsa);

    if (signature !== sigHarusnya) {
        return { valid: false, idUnik: idUnik };
    }

    const hariIni = new Date();
    const tglKadaluarsaObj = new Date(tanggalKadaluarsa);
    const sisa = Math.ceil((tglKadaluarsaObj - hariIni) / (1000 * 60 * 60 * 24));

    return {
        valid: true,
        kadaluarsa: sisa <= 0,
        sisaHari: sisa,
        idUnik: idUnik,
        tanggalKadaluarsa: tglKadaluarsaObj,
    };
}

/**
 * Memvalidasi dan menerapkan kode voucher untuk memperpanjang lisensi.
 * @param {string} kodeVoucher - Kode voucher yang dimasukkan pengguna.
 */
function validasiVoucher(kodeVoucher) {
    const kode = kodeVoucher.replace(/-/g, '').toUpperCase();
    if (kode.length !== 16) {
        alert("Format voucher salah.");
        return false;
    }

    const bekas = JSON.parse(localStorage.getItem(LOKASI_VOUCHER_BEKAS)) || [];
    if (bekas.includes(kode)) {
        alert("Kode voucher ini sudah pernah digunakan!");
        return false;
    }

    const lisensi = JSON.parse(localStorage.getItem(LOKASI_LISENSI));
    const idUnik = lisensi.idUnik;
    const hariStr = kode.substring(0, 3);
    const idChecksum = kode.substring(3, 8);
    const nonce = kode.substring(8, 10);
    const sigVoucher = kode.substring(10, 16);
    const idChecksumHarusnya = buatSignature(idUnik, 5);

    if (idChecksum !== idChecksumHarusnya) {
        alert("Kode voucher ini bukan untuk ID Unik Anda!");
        return false;
    }

    const sigHarusnya = buatSignature(hariStr + idChecksum + nonce, 6);
    if (sigVoucher !== sigHarusnya) {
        alert("Kode voucher tidak valid!");
        return false;
    }

    const hari = parseInt(hariStr, 10);
    let tglLama = new Date(lisensi.tanggalKadaluarsa);
    let basis = new Date() > tglLama ? new Date() : tglLama;
    let tglBaru = new Date(basis);
    tglBaru.setDate(basis.getDate() + hari);
    const tglISO = tglBaru.toISOString();
    const sigLisensiBaru = buatSignature(idUnik + tglISO);

    const lisensiBaru = { ...lisensi, tanggalKadaluarsa: tglISO, signature: sigLisensiBaru };
    localStorage.setItem(LOKASI_LISENSI, JSON.stringify(lisensiBaru));
    bekas.push(kode);
    localStorage.setItem(LOKASI_VOUCHER_BEKAS, JSON.stringify(bekas));
    alert(`SUKSES! Masa aktif ditambah ${hari} hari.`);
    return true;
}