// layout.js

window.globalLayoutDragInfo = null;

// Helper function untuk mencari menu secara rekursif
function _findMenuByIdRecursiveInternal(menusArray, targetMenuId) {
    for (var k = 0; k < menusArray.length; k++) {
        var menuItem = menusArray[k];
        if (menuItem.id === targetMenuId) return menuItem;
        if (menuItem.children && Array.isArray(menuItem.children)) {
            var foundInChildren = _findMenuByIdRecursiveInternal(menuItem.children, targetMenuId);
            if (foundInChildren) return foundInChildren;
        }
    }
    return null;
}

function handleGlobalDocumentMouseMove(event) {
    if (!window.globalLayoutDragInfo || !window.vm.isLayoutEditMode) return;
    event.preventDefault();

    var draggedInfo = window.globalLayoutDragInfo;
    var element = draggedInfo.element;
    var mode = draggedInfo.mode;

    if (mode === 'drag-position') {
        var deltaX = event.clientX - draggedInfo.initialMouseX;
        var deltaY = event.clientY - draggedInfo.initialMouseY;
        var newLeft = draggedInfo.startEleRelativeLeft + deltaX;
        var newTop = draggedInfo.startEleRelativeTop + deltaY;

        element.style.position = 'relative';
        element.style.left = newLeft + 'px';
        element.style.top = newTop + 'px';
    } else if (mode === 'resize-input-width') {
        var deltaXResize = event.clientX - draggedInfo.initialMouseX;
        var newWidth = draggedInfo.initialElementWidth + deltaXResize;
        newWidth = Math.max(50, newWidth);
        element.style.width = newWidth + 'px';
    } else if (mode === 'resize-frame-height') {
        const deltaYResize = event.clientY - draggedInfo.initialMouseY;
        let newHeight = draggedInfo.initialElementHeight + deltaYResize;
        newHeight = Math.max(100, newHeight);
        element.style.height = newHeight + 'px';
    }
}

function handleGlobalDocumentMouseUp(event) {
    if (!window.globalLayoutDragInfo || !window.vm.isLayoutEditMode) return;

    var draggedInfo = window.globalLayoutDragInfo;
    var tabId = draggedInfo.tabId;
    var configKey = draggedInfo.configKey;
    var fieldId = draggedInfo.fieldId;
    var element = draggedInfo.element;
    var mode = draggedInfo.mode;
    var configChangedInThisOperation = false;

    var vueInstance = window.vm;
    var currentTab = vueInstance.openTabs.find(function (t) { return t.id === tabId; });
    if (!currentTab) {
        resetGlobalLayoutDragInfo();
        return;
    }

    var masterMenuToUpdate = _findMenuByIdRecursiveInternal(vueInstance.menus, currentTab.menuConfig.id);

    if (!masterMenuToUpdate || !masterMenuToUpdate.db || !masterMenuToUpdate.db[configKey]) {
        console.error("handleGlobalDocumentMouseUp: Tidak bisa menemukan config menu sumber di vueInstance.menus.");
        resetGlobalLayoutDragInfo();
        return;
    }

    var masterFormConfig = masterMenuToUpdate.db[configKey];
    var tabFormConfig = currentTab.menuConfig.db[configKey];

    if (mode === 'drag-position' && fieldId) {
        element.classList.remove('is-dragging');
        var newKiri = element.style.left || '0px';
        var newAtas = element.style.top || '0px';
        var newPosition = 'relative';

        var masterFieldConfig = masterFormConfig.specificFields.find(function (f) { return f.id === fieldId; });
        var tabFieldConfig = tabFormConfig.specificFields.find(function (f) { return f.id === fieldId; });

        if (masterFieldConfig && tabFieldConfig) {
            if (!masterFieldConfig.layout) masterFieldConfig.layout = {};
            if (!tabFieldConfig.layout) tabFieldConfig.layout = {};

            if (masterFieldConfig.layout.kiri !== newKiri) { masterFieldConfig.layout.kiri = newKiri; configChangedInThisOperation = true; }
            if (masterFieldConfig.layout.atas !== newAtas) { masterFieldConfig.layout.atas = newAtas; configChangedInThisOperation = true; }
            if (masterFieldConfig.layout.position !== newPosition) { masterFieldConfig.layout.position = newPosition; configChangedInThisOperation = true; }

            tabFieldConfig.layout.kiri = newKiri;
            tabFieldConfig.layout.atas = newAtas;
            tabFieldConfig.layout.position = newPosition;
        }
    } else if (mode === 'resize-input-width' && fieldId) {
        element.classList.remove('is-resizing-width');
        var newLebar = element.style.width || 'auto';

        var masterFieldConfigResize = masterFormConfig.specificFields.find(function (f) { return f.id === fieldId; });
        var tabFieldConfigResize = tabFormConfig.specificFields.find(function (f) { return f.id === fieldId; });

        if (masterFieldConfigResize && tabFieldConfigResize) {
            if (!masterFieldConfigResize.layout) masterFieldConfigResize.layout = {};
            if (!tabFieldConfigResize.layout) tabFieldConfigResize.layout = {};

            if (masterFieldConfigResize.layout.lebarInput !== newLebar) {
                masterFieldConfigResize.layout.lebarInput = newLebar;
                configChangedInThisOperation = true;
            }
            tabFieldConfigResize.layout.lebarInput = newLebar;
        }
    } else if (mode === 'resize-frame-height') {
        element.classList.remove('is-resizing-height');
        const finalHeight = parseInt(element.style.height, 10);

        if (masterFormConfig.height !== finalHeight) {
            masterFormConfig.height = finalHeight;
            configChangedInThisOperation = true;
        }
        tabFormConfig.height = finalHeight;
    }

    if (configChangedInThisOperation) {
        vueInstance.unsavedLayoutChanges = true;
        vueInstance.$forceUpdate();
    }
    resetGlobalLayoutDragInfo();
}

function resetGlobalLayoutDragInfo() {
    if (window.globalLayoutDragInfo && window.globalLayoutDragInfo.element) {
        window.globalLayoutDragInfo.element.classList.remove('is-dragging', 'is-resizing-width', 'is-resizing-height');
    }
    window.globalLayoutDragInfo = null;
    document.removeEventListener('mousemove', handleGlobalDocumentMouseMove);
    document.removeEventListener('mouseup', handleGlobalDocumentMouseUp);
}

window.layoutMethods = {
    handleFieldMouseDown: function (event, tabId, configKey, fieldId, interactionTarget) {
        if (!this.isLayoutEditMode || event.button !== 0 || interactionTarget !== 'group') return;

        var currentTab = this.openTabs.find(function (tab) { return tab.id === tabId; });
        if (!currentTab || !currentTab.menuConfig || !currentTab.menuConfig.db || !currentTab.menuConfig.db[configKey]) return;

        var formConfigFromMenu = currentTab.menuConfig.db[configKey];
        if (!formConfigFromMenu || formConfigFromMenu.model !== 'form') return;

        var fieldConfig = formConfigFromMenu.specificFields.find(function (f) { return f.id === fieldId; });
        if (!fieldConfig) return;

        // PERBAIKAN: Cek apakah Shift key ditekan
        if (event.shiftKey) {
            // Shift + Click = Resize width mode
            this.startInputWidthResizeFromField(event, tabId, configKey, fieldId);
        } else {
            // Click biasa = Drag position mode
            this.startFieldDragPosition(event, tabId, configKey, fieldId);
        }
    },

    // Fungsi terpisah untuk drag position
    startFieldDragPosition: function (event, tabId, configKey, fieldId) {
        var currentTab = this.openTabs.find(function (tab) { return tab.id === tabId; });
        var formConfigFromMenu = currentTab.menuConfig.db[configKey];
        var targetManipulatedElement = event.currentTarget;
        var fieldConfig = formConfigFromMenu.specificFields.find(function (f) { return f.id === fieldId; });
        var originalLayout = (fieldConfig.layout) ? JSON.parse(JSON.stringify(fieldConfig.layout)) : {};

        targetManipulatedElement.classList.add('is-dragging');

        window.globalLayoutDragInfo = {
            vueInstance: this,
            tabId: tabId,
            configKey: configKey,
            fieldId: fieldId,
            element: targetManipulatedElement,
            mode: 'drag-position',
            originalLayout: originalLayout,
            initialMouseX: event.clientX,
            initialMouseY: event.clientY,
            startEleRelativeLeft: parseFloat(originalLayout.kiri) || 0,
            startEleRelativeTop: parseFloat(originalLayout.atas) || 0,
        };

        document.addEventListener('mousemove', handleGlobalDocumentMouseMove);
        document.addEventListener('mouseup', handleGlobalDocumentMouseUp);
        event.preventDefault();
    },

    // Fungsi terpisah untuk resize width dari field group
    startInputWidthResizeFromField: function (event, tabId, configKey, fieldId) {
        var currentTab = this.openTabs.find(function (tab) { return tab.id === tabId; });
        var formConfigFromMenu = currentTab.menuConfig.db[configKey];
        var fieldConfig = formConfigFromMenu.specificFields.find(function (f) { return f.id === fieldId; });

        // Cari input wrapper element dalam field group
        var targetElement = event.currentTarget;
        var inputWrapperElement = targetElement.querySelector('.input-wrapper') ||
            targetElement.querySelector('input') ||
            targetElement.querySelector('select') ||
            targetElement.querySelector('textarea') ||
            targetElement; // fallback ke element itu sendiri

        inputWrapperElement.classList.add('is-resizing-width');
        var originalLayout = (fieldConfig.layout) ? JSON.parse(JSON.stringify(fieldConfig.layout)) : {};

        window.globalLayoutDragInfo = {
            vueInstance: this,
            tabId: tabId,
            configKey: configKey,
            fieldId: fieldId,
            element: inputWrapperElement,
            mode: 'resize-input-width',
            originalLayout: originalLayout,
            initialMouseX: event.clientX,
            initialElementWidth: inputWrapperElement.offsetWidth,
        };

        document.addEventListener('mousemove', handleGlobalDocumentMouseMove);
        document.addEventListener('mouseup', handleGlobalDocumentMouseUp);
        event.preventDefault();
    },

    startInputWidthResize: function (event, tabId, configKey, fieldId, inputWrapperElement) {
        if (!this.isLayoutEditMode) return;

        var currentTab = this.openTabs.find(function (tab) { return tab.id === tabId; });
        if (!currentTab || !currentTab.menuConfig || !currentTab.menuConfig.db || !currentTab.menuConfig.db[configKey]) return;

        var formConfigFromMenu = currentTab.menuConfig.db[configKey];
        var fieldConfig = formConfigFromMenu.specificFields.find(function (f) { return f.id === fieldId; });
        if (!fieldConfig) return;

        inputWrapperElement.classList.add('is-resizing-width');
        var originalLayout = (fieldConfig.layout) ? JSON.parse(JSON.stringify(fieldConfig.layout)) : {};

        window.globalLayoutDragInfo = {
            vueInstance: this,
            tabId: tabId,
            configKey: configKey,
            fieldId: fieldId,
            element: inputWrapperElement,
            mode: 'resize-input-width',
            originalLayout: originalLayout,
            initialMouseX: event.clientX,
            initialElementWidth: inputWrapperElement.offsetWidth,
        };

        document.addEventListener('mousemove', handleGlobalDocumentMouseMove);
        document.addEventListener('mouseup', handleGlobalDocumentMouseUp);
    },

    handleFormFrameMouseDown(event, tab, configKey) {
        if (!this.isLayoutEditMode) return;
        event.preventDefault();
        event.stopPropagation();

        const formEditorElement = event.currentTarget.closest('.form-editor');
        if (!formEditorElement) return;

        formEditorElement.classList.add('is-resizing-height');

        window.globalLayoutDragInfo = {
            mode: 'resize-frame-height',
            element: formEditorElement,
            initialMouseY: event.clientY,
            initialElementHeight: formEditorElement.offsetHeight,
            tabId: tab.id,
            configKey: configKey,
        };

        document.addEventListener('mousemove', handleGlobalDocumentMouseMove);
        document.addEventListener('mouseup', handleGlobalDocumentMouseUp);
    },

    initiateSaveLayout: function () {
        var self = this;
        if (!this.unsavedLayoutChanges) {
            return Promise.resolve();
        }
        if (!confirm("Simpan semua perubahan tata letak ke server?")) {
            return Promise.resolve();
        }

        return this.saveAppConfiguration().then(function (success) {
            if (success) {
                self.unsavedLayoutChanges = false;
                alert('Tata letak berhasil disimpan ke server.');
            } else {
                alert('Gagal menyimpan tata letak ke server.');
            }
            return success;
        }).catch(function (error) {
            console.error("Error during initiateSaveLayout:", error);
            alert('Terjadi kesalahan saat menyimpan tata letak.');
            return false;
        });
    },

    resetCurrentTabLayout: function () {
        var self = this;
        if (!this.activeTabId) return Promise.resolve();
        var currentTab = this.openTabs.find(function (tab) { return tab.id === self.activeTabId; });
        if (!currentTab || !currentTab.menuConfig || !currentTab.menuConfig.db) return Promise.resolve();

        var mainConfigKey = currentTab.mainConfigKey;
        var formConfigInTab = (currentTab.menuConfig.db && currentTab.menuConfig.db[mainConfigKey]) ? currentTab.menuConfig.db[mainConfigKey] : null;

        if (!mainConfigKey || !formConfigInTab || formConfigInTab.model !== 'form') {
            alert("Tab aktif bukan form atau tidak memiliki konfigurasi utama untuk layout.");
            return Promise.resolve();
        }
        if (!confirm("Anda yakin ingin mereset tata letak untuk form '" + formConfigInTab.nama + "' ke default?")) {
            return Promise.resolve();
        }

        if (formConfigInTab.specificFields && Array.isArray(formConfigInTab.specificFields)) {
            formConfigInTab.specificFields.forEach(function (field) {
                if (field.layout) {
                    delete field.layout;
                }
            });
        }
        if (formConfigInTab.height) {
            delete formConfigInTab.height;
        }

        var masterMenuToUpdate = _findMenuByIdRecursiveInternal(this.menus, currentTab.menuConfig.id);

        if (masterMenuToUpdate && masterMenuToUpdate.db && masterMenuToUpdate.db[mainConfigKey]) {
            var formConfigInMaster = masterMenuToUpdate.db[mainConfigKey];
            if (formConfigInMaster.specificFields && Array.isArray(formConfigInMaster.specificFields)) {
                formConfigInMaster.specificFields.forEach(function (field) {
                    if (field.layout) {
                        delete field.layout;
                    }
                });
            }
            if (formConfigInMaster.height) {
                delete formConfigInMaster.height;
            }
        }

        this.unsavedLayoutChanges = true;
        this.$forceUpdate();
        alert("Tata letak form telah direset di memori. Klik 'Simpan Layout' untuk menerapkan ke server.");
        return Promise.resolve();
    },

    getFieldStyle: function (tabId, configKey, field) {
        var style = {};
        var currentTab = this.openTabs.find(function (t) { return t.id === tabId; });
        if (!currentTab || !currentTab.menuConfig || !currentTab.menuConfig.db || !currentTab.menuConfig.db[configKey]) return style;

        var fieldCurrentConfig = currentTab.menuConfig.db[configKey].specificFields.find(function (f) { return f.id === field.id; });
        if (fieldCurrentConfig && fieldCurrentConfig.layout) {
            var layout = fieldCurrentConfig.layout;
            if (layout.kiri !== undefined || layout.atas !== undefined) {
                style.position = layout.position || 'relative';
                if (layout.kiri !== undefined) style.left = layout.kiri;
                if (layout.atas !== undefined) style.top = layout.atas;
            }
        }
        return style;
    },

    getInputStyle: function (tabId, configKey, field) {
        var style = {};
        var currentTab = this.openTabs.find(function (t) { return t.id === tabId; });
        if (!currentTab || !currentTab.menuConfig || !currentTab.menuConfig.db || !currentTab.menuConfig.db[configKey]) return style;

        var fieldCurrentConfig = currentTab.menuConfig.db[configKey].specificFields.find(function (f) { return f.id === field.id; });

        if (fieldCurrentConfig && fieldCurrentConfig.layout) {
            var layout = fieldCurrentConfig.layout;
            if (layout.lebarInput !== undefined && layout.lebarInput !== 'auto') {
                style.width = layout.lebarInput;
            }
        }
        return style;
    },

    getFormFrameStyle(tab, configKey) {
        let style = {};
        if (tab && tab.menuConfig && tab.menuConfig.db && tab.menuConfig.db[configKey]) {
            const formConfig = tab.menuConfig.db[configKey];
            if (formConfig.hasOwnProperty('height') && typeof formConfig.height === 'number' && formConfig.height > 0) {
                style.height = formConfig.height + 'px';
            } else {
                style.height = 'auto';
            }
        }
        return style;
    }
};