const { createApp, ref, reactive, computed, onMounted } = Vue;

createApp({
    data() {
        return {
            // Data penjualan utama
            totalPenjualan: 0,
            jumlahTransaksi: 0,
            rataRataPenjualan: 0,
            jumlahItem: 0,

            // Kecenderungan metrik
            persenKecenderunganPenjualan: 8.5,
            persenKecenderunganTransaksi: 12.3,
            persenKecenderunganRataRata: -3.2,
            persenKecenderunganItem: 5.7,

            // Rentang tanggal
            rentangTanggal: '',

            // Grafik
            grafikTrenPenjualan: null,
            grafikKategoriPenjualan: null,

            // Rekap tambahan
            rekapMetodePembayaran: [],
            rekapPenjualanPerItem: [],
            rekapPenjualanPerJenis: [],
            
            // Data baru untuk grid stok barang
            dataStokBarang: [],
            
            // Data baru untuk grid rekap pembelian
            dataPembelian: [],
            
            // Pencarian untuk filter tabel
            searchStokBarang: '',
            searchRekapPembelian: '',
            searchRekapPenjualan: '',
            
            // Sorting untuk tabel
            sortField: {
                stokBarang: 'nama',
                rekapPembelian: 'tanggal',
                rekapPenjualanPerItem: 'nama'
            },
            sortDirection: {
                stokBarang: 'asc',
                rekapPembelian: 'desc',
                rekapPenjualanPerItem: 'asc'
            }
        };
    },

    computed: {
        // Kelas dan ikon kecenderungan untuk penjualan
        kelasKecenderunganPenjualan() {
            return this.persenKecenderunganPenjualan >= 0 ? 'trend-up' : 'trend-down';
        },
        ikonKecenderunganPenjualan() {
            return this.persenKecenderunganPenjualan >= 0 
                ? 'fas fa-arrow-up' 
                : 'fas fa-arrow-down';
        },

        // Kelas dan ikon kecenderungan untuk transaksi
        kelasKecenderunganTransaksi() {
            return this.persenKecenderunganTransaksi >= 0 ? 'trend-up' : 'trend-down';
        },
        ikonKecenderunganTransaksi() {
            return this.persenKecenderunganTransaksi >= 0 
                ? 'fas fa-arrow-up' 
                : 'fas fa-arrow-down';
        },

        // Kelas dan ikon kecenderungan untuk rata-rata
        kelasKecenderunganRataRata() {
            return this.persenKecenderunganRataRata >= 0 ? 'trend-up' :'trend-down';
        },
        ikonKecenderunganRataRata() {
            return this.persenKecenderunganRataRata >= 0 
                ? 'fas fa-arrow-up' 
                : 'fas fa-arrow-down';
        },

        // Kelas dan ikon kecenderungan untuk item
        kelasKecenderunganItem() {
            return this.persenKecenderunganItem >= 0 ? 'trend-up' : 'trend-down';
        },
        ikonKecenderunganItem() {
            return this.persenKecenderunganItem >= 0 
                ? 'fas fa-arrow-up' 
                : 'fas fa-arrow-down';
        },
        
        // Computed untuk Data Stok Barang dengan filter dan sort
        filteredDataStokBarang() {
            let result = [...this.dataStokBarang];
            
            // Filter berdasarkan search
            if (this.searchStokBarang) {
                const search = this.searchStokBarang.toLowerCase();
                result = result.filter(item => 
                    item.nama.toLowerCase().includes(search)
                );
            }
            
            // Sorting
            result.sort((a, b) => {
                const direction = this.sortDirection.stokBarang === 'asc' ? 1 : -1;
                const field = this.sortField.stokBarang;
                
                if (typeof a[field] === 'string') {
                    return direction * a[field].localeCompare(b[field]);
                } else {
                    return direction * (a[field] - b[field]);
                }
            });
            
            return result;
        },
        
        // Computed untuk Rekap Pembelian dengan filter dan sort
        filteredDataPembelian() {
            let result = [...this.dataPembelian];
            
            // Filter berdasarkan search
            if (this.searchRekapPembelian) {
                const search = this.searchRekapPembelian.toLowerCase();
                result = result.filter(item => 
                    item.nama.toLowerCase().includes(search) || 
                    item.relasi.toLowerCase().includes(search)
                );
            }
            
            // Sorting
            result.sort((a, b) => {
                const direction = this.sortDirection.rekapPembelian === 'asc' ? 1 : -1;
                const field = this.sortField.rekapPembelian;
                
                if (field === 'tanggal') {
                    return direction * (new Date(a[field]) - new Date(b[field]));
                } else if (typeof a[field] === 'string') {
                    return direction * a[field].localeCompare(b[field]);
                } else {
                    return direction * (a[field] - b[field]);
                }
            });
            
            return result;
        },
        
        // Computed untuk Rekap Penjualan per Item dengan filter dan sort
        filteredRekapPenjualanPerItem() {
            let result = [...this.rekapPenjualanPerItem];
            
            // Filter berdasarkan search
            if (this.searchRekapPenjualan) {
                const search = this.searchRekapPenjualan.toLowerCase();
                result = result.filter(item => 
                    item.nama.toLowerCase().includes(search)
                );
            }
            
            // Sorting
            result.sort((a, b) => {
                const direction = this.sortDirection.rekapPenjualanPerItem === 'asc' ? 1 : -1;
                const field = this.sortField.rekapPenjualanPerItem;
                
                if (typeof a[field] === 'string') {
                    return direction * a[field].localeCompare(b[field]);
                } else {
                    return direction * (a[field] - b[field]);
                }
            });
            
            return result;
        },
        
        // Computed untuk total stok barang
        totalStokBarang() {
            return this.filteredDataStokBarang.reduce((totals, item) => {
                totals.terima += item.terima || 0;
                totals.jual += item.jual || 0;
                totals.stok += item.stok || 0;
                return totals;
            }, { terima: 0, jual: 0, stok: 0 });
        },
        
        // Computed untuk total rekap pembelian
        totalRekapPembelian() {
            return this.filteredDataPembelian.reduce((total, item) => {
                return total + (item.qty || 0);
            }, 0);
        },
        
        // Computed untuk total rekap penjualan per item
        totalRekapPenjualan() {
            return this.filteredRekapPenjualanPerItem.reduce((totals, item) => {
                totals.qty += item.qty || 0;
                totals.total += item.total || 0;
                return totals;
            }, { qty: 0, total: 0 });
        }
    },

    methods: {
        // Format mata uang Rupiah tanpa desimal
        formatRupiah(angka) {
            if (angka === undefined || angka === null) return 'Rp 0';
            // Bulatkan angka terlebih dahulu
            angka = Math.round(angka);
            return 'Rp ' + new Intl.NumberFormat('id-ID').format(angka);
        },
        
        // Toggle sorting untuk tabel
        toggleSort(tabel, field) {
            if (this.sortField[tabel] === field) {
                // Ubah arah sorting jika field yang sama
                this.sortDirection[tabel] = this.sortDirection[tabel] === 'asc' ? 'desc' : 'asc';
            } else {
                // Set field baru dengan arah default
                this.sortField[tabel] = field;
                this.sortDirection[tabel] = 'asc';
            }
        },
        
        // Mendapatkan class untuk icon sorting
        getSortIconClass(tabel, field) {
            if (this.sortField[tabel] !== field) {
                return 'fas fa-sort text-muted';
            }
            return this.sortDirection[tabel] === 'asc' 
                ? 'fas fa-sort-up text-primary' 
                : 'fas fa-sort-down text-primary';
        },

        // Hitung total penjualan dengan rumus baru
        hitungPenjualanItem(item) {
            const qty = parseFloat(item.QTT || 0);
            const harga = parseFloat(item.HARGA || 0);
            const diskRupiah = parseFloat(item.DISKRUPIAH || 0);
            const diskP = parseFloat(item.DISKP || 0);

            // Rumus: qtt * (harga - diskrupiah) * (100 - diskp) / 100
            return qty * (harga - diskRupiah) * (100 - diskP) / 100;
        },

        // Ambil data penjualan
        async ambilDataPenjualan(tanggalMulai, tanggalSelesai) {
            try {
                console.log(`Memuat data penjualan untuk rentang: ${tanggalMulai} - ${tanggalSelesai}`);
                
                // Gunakan fungsi ambil dari crud.js
                const dataPenjualan = await this.panggilAjax('penjualan', 'TANGGAL', tanggalMulai, tanggalSelesai);
                
                // Filter data yang tidak dihapus
                const dataTerfilter = Array.isArray(dataPenjualan) 
                    ? dataPenjualan.filter(item => item.DEL !== '#')
                    : [];
                
                console.log(`Data penjualan terfilter: ${dataTerfilter.length} baris`);
                
                // Perbarui metrik dashboard
                this.perbaruiMetrik(dataTerfilter);
                this.perbaruiGrafik(dataTerfilter);
                this.perbaruiRekap(dataTerfilter);
                
                // Memanggil fungsi untuk memperbarui data grid stok barang dan rekap pembelian
                this.perbaruiDataBarang();
                this.perbaruiRekapPembelian(tanggalMulai, tanggalSelesai);
            } catch (error) {
                console.error('Kesalahan memuat data:', error);
                alert('Terjadi kesalahan saat memuat data dashboard: ' + error.message);
            }
        },

        // Panggil AJAX secara manual dengan penanganan error yang lebih baik
        panggilAjax(tabel, field, value1, value2) {
            return new Promise((resolve, reject) => {
                if (!value2) value2 = value1;

                console.log(`Memanggil AJAX: table=${tabel}, field=${field}, value=[${value1}, ${value2}]`);

                $.ajax({
                    type: "POST",
                    url: "crud.php",
                    data: {
                        action: 'ambil',
                        table: tabel,
                        field: field,
                        value: [value1, value2]
                    },
                    success: (response) => {
                        try {
                            // alert(tabel+ field+ value1+ value2 +JSON.stringify( response));
                            console.log(`Respons mentah untuk ${tabel}:`, response);
                            
                            // Periksa apakah respons kosong
                            if (!response || response.trim() === '') {
                                console.warn(`Respons kosong dari server untuk tabel ${tabel}`);
                                resolve([]);
                                return;
                            }
                            
                            // Coba parse sebagai JSON
                            let dataParsed;
                            if (typeof response === 'string') {
                                try {
                                    dataParsed = JSON.parse(response);
                                } catch (parseError) {
                                    console.error(`Error parsing JSON untuk tabel ${tabel}:`, parseError);
                                    console.error('Respons yang tidak bisa di-parse:', response);
                                    // Kembalikan array kosong jika parsing gagal
                                    resolve([]);
                                    return;
                                }
                            } else {
                                dataParsed = response;
                            }
                            
                            // Periksa apakah hasilnya adalah array
                            if (!Array.isArray(dataParsed)) {
                                console.warn(`Respons bukan array untuk tabel ${tabel}:`, dataParsed);
                                resolve([]);
                                return;
                            }
                            
                            resolve(dataParsed);
                        } catch (error) {
                            console.error(`Error umum dalam handler success untuk tabel ${tabel}:`, error);
                            reject(error);
                        }
                    },
                    error: (xhr, status, error) => {
                        console.error(`Error AJAX untuk tabel ${tabel}:`, {xhr, status, error});
                        reject(error);
                    }
                });
            });
        },

        // Perbarui metrik dashboard
        perbaruiMetrik(dataPenjualan) {
            // Hitung total penjualan menggunakan rumus baru
            this.totalPenjualan = dataPenjualan.reduce((total, item) => {
                return total + this.hitungPenjualanItem(item);
            }, 0);
            
            // Hitung transaksi unik
            const transaksi = new Set(
                dataPenjualan.map(item => item.NOMOR).filter(Boolean)
            );
            this.jumlahTransaksi = transaksi.size;
            
            // Hitung total item
            this.jumlahItem = dataPenjualan.reduce((total, item) => {
                const qty = parseInt(item.QTT || 0);
                return isNaN(qty) ? total : total + qty;
            }, 0);
            
            // Hitung rata-rata penjualan (dibulatkan)
            this.rataRataPenjualan = this.jumlahTransaksi > 0 
                ? Math.round(this.totalPenjualan / this.jumlahTransaksi) 
                : 0;
            
            // TODO: Implementasi untuk menghitung persentase perbandingan dengan periode sebelumnya
            // Saat ini nilai persentase masih berupa data dummy
        },

        // Perbarui rekap tambahan
        perbaruiRekap(dataPenjualan) {
            // Rekap Metode Pembayaran
            const metodePembayaran = {};
            
            // Rekap Penjualan per Item
            const penjualanPerItem = {};
            
            // Rekap Penjualan per Jenis
            const penjualanPerJenis = {};

            // Proses data penjualan
            dataPenjualan.forEach(item => {
                // Jika item memiliki nilai PAYMENT, masukkan ke rekap metode pembayaran
                if (item.PAYMENT) {
                    const metodeBayar = item.NAMA || 'Tidak Diketahui';
                    const nilaiPembayaran = parseFloat(item.HARGA || 0);
                    
                    if (!isNaN(nilaiPembayaran)) {
                        metodePembayaran[metodeBayar] = (metodePembayaran[metodeBayar] || 0) + nilaiPembayaran;
                    }
                } else {
                    // Proses item penjualan normal (bukan record pembayaran)
                    // Rekap Penjualan per Item
                    const namaItem = item.NAMA || 'Tidak Bernama';
                    const qty = parseFloat(item.QTT || 0);
                    const nilaiPenjualan = this.hitungPenjualanItem(item);
                    
                    if (!penjualanPerItem[namaItem]) {
                        penjualanPerItem[namaItem] = { qty: 0, total: 0 };
                    }
                    
                    penjualanPerItem[namaItem].qty += qty;
                    penjualanPerItem[namaItem].total += nilaiPenjualan;

                    // Rekap Penjualan per Jenis
                    const jenis = item.JENIS || 'Tidak Berkategori';
                    penjualanPerJenis[jenis] = (penjualanPerJenis[jenis] || 0) + nilaiPenjualan;
                }
            });

            // Konversi ke array dan urutkan
            this.rekapMetodePembayaran = Object.entries(metodePembayaran)
                .map(([metode, total]) => ({ metode, total }))
                .sort((a, b) => b.total - a.total);

            this.rekapPenjualanPerItem = Object.entries(penjualanPerItem)
                .map(([nama, data]) => ({ nama, ...data }))
                .sort((a, b) => b.qty - a.qty);

            this.rekapPenjualanPerJenis = Object.entries(penjualanPerJenis)
                .map(([jenis, omzet]) => ({ jenis, omzet }))
                .sort((a, b) => b.omzet - a.omzet);
        },
        
        // Perbarui data untuk grid stok barang
        async perbaruiDataBarang() {
            try {
                console.log("Mulai memuat data barang");
                
                // Ambil data barang dengan stok
                const dataBarang = await this.panggilAjax('barang', 'KODE', '0', '999999');
                console.log("Data barang diterima:", dataBarang);
                
                // Ambil semua data penerimaan untuk menghitung total terima
                const dataPenerimaan = await this.panggilAjax('pembelian', 'BARANG', '0', '999999');
                console.log("Data penerimaan diterima:", dataPenerimaan);
                
                // Ambil semua data penjualan untuk menghitung total jual
                const dataPenjualan = await this.panggilAjax('penjualan', 'BARANG', '0', '999999');
                console.log("Data penjualan diterima:", dataPenjualan);
                
                // Jika data kosong, tetap lanjutkan dengan array kosong
                const dataStok = [];
                
                // Membuat kamus untuk total terima per barang
                const totalTerimaPerBarang = {};
                if (Array.isArray(dataPenerimaan)) {
                    dataPenerimaan.forEach(item => {
                        // Gunakan kolom BARANG sebagai kode barang
                        const kode = item.BARANG || '';
                        const qty = parseFloat(item.QTT || 0);
                        
                        if (kode && !isNaN(qty)) {
                            totalTerimaPerBarang[kode] = (totalTerimaPerBarang[kode] || 0) + qty;
                        }
                    });
                }
                console.log("Total terima per barang:", totalTerimaPerBarang);
                
                // Membuat kamus untuk total jual per barang
                const totalJualPerBarang = {};
                if (Array.isArray(dataPenjualan)) {
                    dataPenjualan.forEach(item => {
                        // Gunakan kolom BARANG sebagai kode barang
                        const kode = item.BARANG || '';
                        const qty = parseFloat(item.QTT || 0);
                        
                        if (kode && !isNaN(qty)) {
                            totalJualPerBarang[kode] = (totalJualPerBarang[kode] || 0) + qty;
                        }
                    });
                }
                console.log("Total jual per barang:", totalJualPerBarang);
                
                // Proses data barang
                if (Array.isArray(dataBarang)) {
                    dataBarang.forEach(item => {
                        const kode = item.KODE || '';
                        const nama = item.NAMA || 'Tidak Ada Nama';
                        const stok = parseFloat(item.STOKC || 0);
                        
                        // Ambil jumlah terima dan jual dari kamus
                        const terima = totalTerimaPerBarang[kode] || 0;
                        const jual = totalJualPerBarang[kode] || 0;
                        
                        // Tambahkan ke array dataStok
                        dataStok.push({
                            kode,
                            nama,
                            terima,
                            jual,
                            stok
                        });
                    });
                }
                
                // Urutkan berdasarkan nama
                dataStok.sort((a, b) => a.nama.localeCompare(b.nama));
                console.log("Data stok barang final:", dataStok);
                
                // Update state
                this.dataStokBarang = dataStok;
                
            } catch (error) {
                console.error('Kesalahan memuat data barang:', error);
                // Jangan tampilkan alert karena akan mengganggu
                this.dataStokBarang = [];
            }
        },
        
        // Perbarui data untuk grid rekap pembelian
        async perbaruiRekapPembelian(tanggalMulai, tanggalSelesai) {
            try {
                // Jika tanggal tidak disediakan, gunakan rentang dari daterangepicker
                if (!tanggalMulai || !tanggalSelesai) {
                    const rentangTanggal = $('#reportrange').data('daterangepicker');
                    tanggalMulai = rentangTanggal.startDate.format('YYYY-MM-DD');
                    tanggalSelesai = rentangTanggal.endDate.format('YYYY-MM-DD');
                }
                
                console.log(`Memuat data pembelian untuk rentang: ${tanggalMulai} - ${tanggalSelesai}`);
                
                // Ambil data pembelian untuk rentang tanggal yang dipilih
                // Perhatikan: Berdasarkan struktur tabel diberikan, seharusnya kita menggunakan tabel 'pembelian'
                // bukan 'pembelian', tapi kita akan mencoba keduanya
                let dataPembelian = [];
                
                try {
                    dataPembelian = await this.panggilAjax('pembelian', 'TANGGAL', tanggalMulai, tanggalSelesai);
                    console.log("Data dari tabel 'pembelian':", dataPembelian);
                } catch (error) {
                    console.log("Gagal mengambil dari tabel 'pembelian', mencoba 'pembelian'");
                    try {
                        dataPembelian = await this.panggilAjax('pembelian', 'TANGGAL', tanggalMulai, tanggalSelesai);
                        console.log("Data dari tabel 'pembelian':", dataPembelian);
                    } catch (error2) {
                        console.error("Gagal mengambil data dari kedua tabel:", error2);
                    }
                }
                
                // Proses data untuk grid rekap pembelian
                const rekapPembelian = [];
                
                if (Array.isArray(dataPembelian)) {
                    dataPembelian.forEach(item => {
                        // Filter data yang sudah dihapus
                        if (item.DEL === '#') return;
                        
                        const pembelian = {
                            tanggal: item.TANGGAL || '',
                            relasi: item.RELASI || 'Tidak Ada Relasi',
                            nama: item.NAMA || 'Tidak Ada Nama',
                            qty: parseFloat(item.QTT || 0)
                        };
                        
                        if (!isNaN(pembelian.qty) && pembelian.qty > 0) {
                            rekapPembelian.push(pembelian);
                        }
                    });
                }
                
                // Urutkan berdasarkan tanggal terbaru
                rekapPembelian.sort((a, b) => {
                    // Pastikan kita memiliki tanggal yang valid
                    const dateA = a.tanggal ? new Date(a.tanggal) : new Date(0);
                    const dateB = b.tanggal ? new Date(b.tanggal) : new Date(0);
                    return dateB - dateA;
                });
                
                console.log("Rekap pembelian final:", rekapPembelian);
                
                // Update state
                this.dataPembelian = rekapPembelian;
                
            } catch (error) {
                console.error('Kesalahan memuat data pembelian:', error);
                // Jangan tampilkan alert karena akan mengganggu
                this.dataPembelian = [];
            }
        },

        // Perbarui grafik
        perbaruiGrafik(dataPenjualan) {
            // Hancurkan grafik sebelumnya
            this.hancurkanGrafik();
            
            // Perbarui grafik tren penjualan
            this.perbaruiGrafikTrenPenjualan(dataPenjualan);
            
            // Perbarui grafik kategori penjualan
            this.perbaruiGrafikKategori(dataPenjualan);
        },

        // Hancurkan grafik sebelumnya
        hancurkanGrafik() {
            const daftarGrafik = [
                'grafikTrenPenjualan', 
                'grafikKategoriPenjualan'
            ];
            
            daftarGrafik.forEach(namaGrafik => {
                const grafik = Chart.getChart(namaGrafik);
                if (grafik) grafik.destroy();
            });
        },

        // Perbarui grafik tren penjualan
        perbaruiGrafikTrenPenjualan(dataPenjualan) {
            // Kelompokkan data berdasarkan tanggal
            const dataBerdasarkanTanggal = {};
            
            dataPenjualan.forEach(item => {
                const tanggal = item.TANGGAL;
                const jumlah = this.hitungPenjualanItem(item);
                
                if (!isNaN(jumlah) && tanggal) {
                    dataBerdasarkanTanggal[tanggal] = 
                        (dataBerdasarkanTanggal[tanggal] || 0) + jumlah;
                }
            });
            
            // Siapkan data untuk grafik
            const label = Object.keys(dataBerdasarkanTanggal).sort();
            const data = label.map(tgl => dataBerdasarkanTanggal[tgl]);
            
            // Buat grafik
            const ctx = document.getElementById('grafikTrenPenjualan').getContext('2d');
            
            this.grafikTrenPenjualan = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: label,
                    datasets: [{
                        label: 'Total Penjualan',
                        data: data,
                        backgroundColor: 'rgba(13, 110, 253, 0.1)',
                        borderColor: 'rgba(13, 110, 253, 1)',
                        borderWidth: 2,
                        tension: 0.4,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { display: false },
                        tooltip: {
                            callbacks: {
                                label: (konteks) => this.formatRupiah(konteks.raw)
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                callback: (nilai) => this.formatRupiah(nilai)
                            }
                        }
                    }
                }
            });
        },

        // Perbarui grafik kategori penjualan
        perbaruiGrafikKategori(dataPenjualan) {
            // Kelompokkan data berdasarkan kategori
            const dataKategori = {};
            
            dataPenjualan.forEach(item => {
                const kategori = item.JENIS || 'Tidak Berkategori';
                const jumlah = this.hitungPenjualanItem(item);
                
                if (!isNaN(jumlah)) {
                    dataKategori[kategori] = (dataKategori[kategori] || 0) + jumlah;
                }
            });
            
            // Siapkan data untuk grafik
            const label = Object.keys(dataKategori);
            const data = label.map(kategori => dataKategori[kategori]);
            

                // Warna untuk grafik
const warna = [
    'rgba(13, 110, 253, 0.7)',
    'rgba(25, 135, 84, 0.7)',
    'rgba(255, 193, 7, 0.7)',
    'rgba(220, 53, 69, 0.7)',
    'rgba(111, 66, 193, 0.7)'
];

// Buat grafik
const ctx = document.getElementById('grafikKategoriPenjualan').getContext('2d');

this.grafikKategoriPenjualan = new Chart(ctx, {
    type: 'pie',
    data: {
        labels: label,
        datasets: [{
            data: data,
            backgroundColor: warna.slice(0, label.length)
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                position: 'right',
                labels: {
                    boxWidth: 15,
                    padding: 10
                }
            },
            tooltip: {
                callbacks: {
                    label: (konteks) => {
                        const value = konteks.raw;
                        const total = konteks.dataset.data.reduce((a, b) => a + b, 0);
                        const persentase = ((value / total) * 100).toFixed(1);
                        return `${konteks.label}: ${this.formatRupiah(value)} (${persentase}%)`;
                    }
                }
            }
        }
    }
});
},


// Inisialisasi pemilih rentang tanggal
inisialisasiRentangTanggal() {
    const self = this;
    const mulai = moment().subtract(29, 'days');
    const selesai = moment();

    function cb(mulai, selesai) {
        self.rentangTanggal = `${mulai.format('D MMM YYYY')} - ${selesai.format('D MMM YYYY')}`;
        self.ambilDataPenjualan(mulai.format('YYYY-MM-DD'), selesai.format('YYYY-MM-DD'));
    }

    function getDayNameIndonesian(date) {
        const days = ['Minggu', 'Senin', 'Selasa', 'Rabu', 'Kamis', 'Jumat', 'Sabtu'];
        return days[date.day()];
    }

    // Buat objek rentang khusus untuk hari-hari sebelumnya
    const customRanges = {};

    // Tambahkan rentang untuk hari ini dan 6 hari sebelumnya
    for (let i = 0; i <= 6; i++) {
        const tanggal = moment().subtract(i, 'days');
        const namaHari = getDayNameIndonesian(tanggal);
        const label = i === 0 
            ? `Hari Ini (${namaHari})` 
            : `H-${i} (${namaHari})`;
        
        customRanges[label] = [tanggal, tanggal];
    }

    // Tambahkan rentang yang sudah ada sebelumnya
    Object.assign(customRanges, {
        '7 Hari Terakhir': [moment().subtract(6, 'days'), moment()],
        '30 Hari Terakhir': [moment().subtract(29, 'days'), moment()],
        'Bulan Ini': [moment().startOf('month'), moment().endOf('month')],
        'Bulan Lalu': [moment().subtract(1, 'month').startOf('month'), moment().subtract(1, 'month').endOf('month')]
    });

    $('#reportrange').daterangepicker({
        startDate: mulai,
        endDate: selesai,
        ranges: customRanges,
        locale: {
            format: 'D MMM YYYY',
            applyLabel: 'Pilih',
            cancelLabel: 'Batal',
            customRangeLabel: 'Rentang Kustom'
        }
    }, cb);

    // Panggilan awal
    cb(mulai, selesai);
},

// Perbarui dashboard
perbarui() {
const rentangTanggal = $('#reportrange').data('daterangepicker');
const tanggalMulai = rentangTanggal.startDate.format('YYYY-MM-DD');
const tanggalSelesai = rentangTanggal.endDate.format('YYYY-MM-DD');
this.ambilDataPenjualan(tanggalMulai, tanggalSelesai);
}
},

mounted() {
// Inisialisasi pemilih rentang tanggal saat komponen dimuat
this.inisialisasiRentangTanggal();
}
}).mount('#app');


