<?php
header("Access-Control-Allow-Origin: *"); // Mengizinkan semua domain. Ganti '*' dengan domain spesifik untuk keamanan lebih.
header("Access-Control-Allow-Methods: POST, GET, OPTIONS"); // Mengizinkan metode request yang digunakan.
header("Access-Control-Allow-Headers: Content-Type, X-Requested-With"); // Mengizinkan header tertentu.
header('Content-Type: application/json');

class DatabaseHandler {
    private $host = 'localhost';
    private $dbname = 'dmoxnpnp_pivot';
    private $username = 'dmoxnpnp_pos';
    private $password = 'jakarta183';
    private $conn = null;
    
    public function __construct() {
        try {
            $this->conn = new PDO(
                "mysql:host=$this->host;dbname=$this->dbname;charset=utf8mb4",
                $this->username,
                $this->password,
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
        } catch (PDOException $e) {
            die(json_encode([
                'success' => false,
                'message' => 'Database connection failed: ' . $e->getMessage()
            ]));
        }
    }
    
    public function processData($data) {
        $results = [
            'total_record' => count($data),
            'inserted' => 0,
            'updated' => 0,
            'deleted' => 0,
            'errors' => []
        ];
        
        try {
            $this->conn->beginTransaction();
            
            foreach ($data as $record) {
                // Check if noit is set in the record
                if (!isset($record['NOIT'])) {
                    $results['errors'][] = "Missing NOIT in record: " . json_encode($record);
                    continue;
                }
                
                $noit = $record['NOIT'];
                
                // Check if delete flag is set to '*'
                if (isset($record['DEL']) && $record['DEL'] === '*') {
                    $this->deleteRecord($noit);
                    $results['deleted']++;
                    continue;
                }
                
                // Check if record exists
                if ($this->recordExists($noit)) {
                    // Update existing record
                    $this->updateRecord($record);
                    $results['updated']++;
                } else {
                    // Insert new record
                    $this->insertRecord($record);
                    $results['inserted']++;
                }
                
                // Update barang after insert/update
                $this->updateBarang($record);
            }
            
            $this->conn->commit();
            $results['success'] = true;
            
        } catch (Exception $e) {
            $this->conn->rollBack();
            $results['success'] = false;
            $results['message'] = 'Error processing data: ' . $e->getMessage();
        }
        
        return $results;
    }
    
    private function recordExists($noit) {
        $stmt = $this->conn->prepare("SELECT COUNT(*) FROM penjualan WHERE noit = :noit");
        $stmt->bindParam(':noit', $noit);
        $stmt->execute();
        return $stmt->fetchColumn() > 0;
    }
    
    private function deleteRecord($noit) {
        $stmt = $this->conn->prepare("DELETE FROM penjualan WHERE noit = :noit");
        $stmt->bindParam(':noit', $noit);
        $stmt->execute();
    }
    
    private function updateRecord($record) {
        $sql = "UPDATE penjualan SET 
                nomor = :nomor,
                tanggal = :tanggal,
                jumlah = :jumlah,
                payment = :payment,
                barang = :barang,
                nama = :nama,
                qtt = :qtt,
                harga = :harga,
                diskp = :diskp,
                diskrupiah = :diskrupiah,
                user = :user,
                tglo = :tglo,
                edc = :edc,
                stok = :stok,
                top = :top,
                del = :del,
                jenis = :group
                WHERE noit = :noit";
                
        $stmt = $this->conn->prepare($sql);
        
        $this->bindRecordParams($stmt, $record);
        $stmt->execute();
    }
    
    private function insertRecord($record) {
        $sql = "INSERT INTO penjualan (
                noit, nomor, tanggal, jumlah, payment, barang, nama, qtt,del, 
                harga, diskp, diskrupiah, user, tglo, edc, stok, top, jenis
            ) VALUES (
                :noit, :nomor, :tanggal, :jumlah, :payment, :barang, :nama, :qtt,:del,
                :harga, :diskp, :diskrupiah, :user, :tglo, :edc, :stok, :top, :group
            )";
            
        $stmt = $this->conn->prepare($sql);
        
        $this->bindRecordParams($stmt, $record);
        $stmt->execute();
    }
    
    private function bindRecordParams($stmt, $record) {
        $stmt->bindParam(':noit', $record['NOIT']);
        $stmt->bindParam(':nomor', $record['NOMOR']);
        
        // Convert date format if needed
        $tanggal = isset($record['TANGGAL']) ? date('Y-m-d', strtotime($record['TANGGAL'])) : null;
        $stmt->bindParam(':tanggal', $tanggal);
        
        $stmt->bindParam(':jumlah', $record['JUMLAH']);
        $stmt->bindParam(':payment', $record['PAYMENT']);
        $stmt->bindParam(':barang', $record['BARANG']);
        $stmt->bindParam(':nama', $record['NAMA']);
        $stmt->bindParam(':qtt', $record['QTT']);
        $stmt->bindParam(':harga', $record['HARGA']);
        $stmt->bindParam(':diskp', $record['DISKP']);
        $stmt->bindParam(':diskrupiah', $record['DISKRUPIAH']);
        $stmt->bindParam(':user', $record['USER']);
        
        // Convert datetime format if needed
        $tglo = isset($record['TGLO']) ? date('Y-m-d H:i:s', strtotime($record['TGLO'])) : null;
        $stmt->bindParam(':tglo', $tglo);
        
        $stmt->bindParam(':edc', $record['EDC']);
        $stmt->bindParam(':stok', $record['STOK']);
        $stmt->bindParam(':top', $record['TOP']);
        $stmt->bindParam(':del', $record['DEL']);
        $stmt->bindParam(':group', $record['GROUP']);
    }
    
    // New method to update barang table
    private function updateBarang($record) {
        // Check if barang exists
        $stmt = $this->conn->prepare("SELECT COUNT(*) FROM barang WHERE kode = :kode");
        $stmt->bindParam(':kode', $record['BARANG']);
        $stmt->execute();
        $exists = $stmt->fetchColumn() > 0;
        
        if (!$exists) {
            // Insert new barang if not found
            $sql = "INSERT INTO barang (
                    kode, nama, top, jenis, stok, stokc, harga, discp, discr
                ) VALUES (
                    :kode, :nama, :top, :jenis, :stok, :stokc, :harga, 0, 0
                )";
                
            $stmt = $this->conn->prepare($sql);
            $stmt->bindParam(':kode', $record['BARANG']);
            $stmt->bindParam(':nama', $record['NAMA']);
            $stmt->bindParam(':top', $record['TOP']);
            $stmt->bindParam(':jenis', $record['GROUP']);
            
            // Initialize stok with negative QTT
            $negativeQtt = -floatval($record['QTT']);
            $stmt->bindParam(':stok', $negativeQtt);
            $stmt->bindParam(':stokc', $record['STOK']);
            $stmt->bindParam(':harga', $record['HARGA']);
            $stmt->execute();
        } else {
            // Update existing barang
            if (isset($record['DEL']) && $record['DEL'] === '#') {
                // If DEL is '#', add QTT to stok
                $sql = "UPDATE barang SET 
                        stok = stok + :qtt,
                        stokc = :stokc
                        WHERE kode = :kode";
            } else {
                // Otherwise, subtract QTT from stok
                $sql = "UPDATE barang SET 
                        stok = stok - :qtt,
                        stokc = :stokc
                        WHERE kode = :kode";
            }
            
            $stmt = $this->conn->prepare($sql);
            $stmt->bindParam(':qtt', $record['QTT']);
            $stmt->bindParam(':stokc', $record['STOK']);
            $stmt->bindParam(':kode', $record['BARANG']);
            $stmt->execute();
        }
    }
}

// Main execution
try {
    $jsonData = file_get_contents('php://input');
    $data = json_decode($jsonData, true);
    
    // Check if data is valid
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Invalid JSON data: ' . json_last_error_msg());
    }
    
    // Handle empty data
    if (empty($data) || !is_array($data)) {
        echo json_encode([
            'success' => false,
            'message' => 'No valid data provided'
        ]);
        exit;
    }
    
    // Process data
    $db = new DatabaseHandler();
    $result = $db->processData($data);
    
    echo json_encode($result);
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>
